document.addEventListener('DOMContentLoaded', () => {
    const mainHeader = document.querySelector('.main-header');
    const searchButton = document.querySelector('.search-btn');
    const tempModalContainer = document.getElementById('temp-modal-container');

    if (mainHeader) {
        const headerObserver = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (!entry.isIntersecting) {
                    mainHeader.classList.add('scrolled');
                } else {
                    mainHeader.classList.remove('scrolled');
                }
            });
        }, { rootMargin: '-10px 0px 0px 0px' });

        headerObserver.observe(document.body);
    }

    function showTemporaryModal(message) {
        if (!tempModalContainer) return;

        const modal = document.createElement('div');
        modal.className = 'temp-modal';
        modal.innerHTML = `
            <div class="temp-modal-content">
                <p>${message}</p>
            </div>
        `;
        tempModalContainer.appendChild(modal);

        setTimeout(() => {
            modal.classList.add('show');
        }, 10);

        setTimeout(() => {
            modal.classList.remove('show');
            modal.addEventListener('transitionend', () => modal.remove(), { once: true });
        }, 2000);
    }

    if (searchButton) {
        searchButton.addEventListener('click', () => {
            const searchInput = searchButton.previousElementSibling;
            if (searchInput && searchInput.value.trim() !== '') {
                showTemporaryModal(`Searching for "${searchInput.value}"...`);
                searchInput.value = '';
            } else {
                showTemporaryModal('Please enter a search term.');
            }
        });
    }

    const sliderWrapper = document.querySelector('.slider-wrapper');
    const sliderItems = document.querySelectorAll('.slider-item');
    const prevBtn = document.querySelector('.prev-btn');
    const nextBtn = document.querySelector('.next-btn');
    const sliderIndicators = document.querySelector('.slider-indicators');

    if (sliderWrapper && sliderItems.length > 0 && prevBtn && nextBtn && sliderIndicators) {
        let currentIndex = 0;
        let intervalId;
        const slideWidth = sliderItems[0].clientWidth;

        function createIndicators() {
            sliderItems.forEach((_, index) => {
                const dot = document.createElement('div');
                dot.classList.add('indicator-dot');
                if (index === 0) dot.classList.add('active');
                dot.addEventListener('click', () => showSlide(index));
                sliderIndicators.appendChild(dot);
            });
        }

        function updateIndicators() {
            document.querySelectorAll('.indicator-dot').forEach((dot, index) => {
                if (index === currentIndex) {
                    dot.classList.add('active');
                } else {
                    dot.classList.remove('active');
                }
            });
        }

        function showSlide(index) {
            if (index < 0) {
                currentIndex = sliderItems.length - 1;
            } else if (index >= sliderItems.length) {
                currentIndex = 0;
            } else {
                currentIndex = index;
            }
            sliderWrapper.style.transform = `translateX(-${currentIndex * 100}%)`;
            updateIndicators();
        }

        function nextSlide() {
            showSlide(currentIndex + 1);
        }

        function prevSlide() {
            showSlide(currentIndex - 1);
        }

        function startAutoSlide() {
            stopAutoSlide();
            intervalId = setInterval(() => {
                requestAnimationFrame(nextSlide);
            }, 3500);
        }

        function stopAutoSlide() {
            clearInterval(intervalId);
        }

        createIndicators();
        showSlide(0);
        startAutoSlide();

        prevBtn.addEventListener('click', () => {
            stopAutoSlide();
            prevSlide();
            startAutoSlide();
        });

        nextBtn.addEventListener('click', () => {
            stopAutoSlide();
            nextSlide();
            startAutoSlide();
        });

        sliderWrapper.addEventListener('mouseenter', stopAutoSlide);
        sliderWrapper.addEventListener('mouseleave', startAutoSlide);

        window.addEventListener('resize', () => {
            sliderWrapper.style.transition = 'none';
            showSlide(currentIndex);
            setTimeout(() => {
                sliderWrapper.style.transition = 'transform 0.5s ease-in-out';
            }, 50);
        });
    }

    const animateSections = document.querySelectorAll('.hero-section, #popular-games, #game-detail, #user-reviews, #related-games, .main-footer');

    const sectionObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate-in');
                observer.unobserve(entry.target);
            }
        });
    }, { threshold: 0.1 });

    animateSections.forEach(section => {
        sectionObserver.observe(section);
    });
});